/*global define*/

define(["lodash", "src/utils", "src/math/Mat3", "src/math/Vec2"],
function(lodash, utils, mat3) {
	"use strict";

	var arrayPush = Array.prototype.push;

	function fill (value, length) {
		var result = [],
			index = -1;

		while (++index < length) result[index] = value;

		return result;
	}

	/**
	 * Test if the argument is a valid handle ref.
	 * Note: Don't rely on simple falsey test because zero is falsey and we use it as a valid handle ref.
	 */
	function isValidRef (ref) {
		return typeof ref === "number" && ref >= 0;
	}

	/**
	 * Test if the matrix is defined.
	 * Note: Don't rely on simple falsey test because null is falsey and we use to identify singular and missing matrices.
	 */
	function isUndefinedMatrix (mat) {
		return typeof mat === "undefined";
	}

	function invertOrNull (mat) {
		var result;
		try {
			result = mat3.invert(mat);
		} catch (e) {
			result = null;
		}
		return result;
	}

	function enableRelativeTo (kRelativeTo, tree) {

		if (kRelativeTo === "puppet") {

			var matLayerParent_Puppet = tree.aMatLayerParent_Puppet[0],
				matPuppet_LayerParent = tree.aMatPuppet_LayerParent[0];

			tree.aMatLayerParent_Puppet[0] = null;
			tree.aMatPuppet_LayerParent[0] = null;

			return function () {
				tree.aMatLayerParent_Puppet[0] = matLayerParent_Puppet;
				tree.aMatPuppet_LayerParent[0] = matPuppet_LayerParent;
			};

		}

		if (kRelativeTo === "layer") {
			return function () {};
		}

		utils.assert(false, "enableRelativeTo(): unknown frame.");
	}

	function getLocalMatrixAtRest (handle, result0) {
		return handle.getMatrixAtRestRelativeToParent(result0);
	}

	function getAccumulatedHandleMatrix (tree, kRootFrame, handleRef, getMatrixFuncName, result0) {
		var disableRelativeTo = enableRelativeTo(kRootFrame, tree),
			mat_Handle = mat3.identity(result0);

		// walk upward (to root) to accumulate matrix
		while ( isValidRef(handleRef) ) {
			var handle = tree.aHandle[handleRef],
				matLayerParent_Puppet = tree.aMatLayerParent_Puppet[handleRef],
				handleMatrix = handle[getMatrixFuncName].call(handle);
			mat3.multiply(handleMatrix, mat_Handle, mat_Handle);
			if (matLayerParent_Puppet) mat3.multiply(matLayerParent_Puppet, mat_Handle, mat_Handle);
			handleRef = tree.aParentRef[handleRef];
		}

		disableRelativeTo();
		return mat_Handle;
	}

	function getAccumulatedTree (kRootFrame, getMatrixRelativeToParent) {
		/*jshint validthis: true*/
		var disableRelativeTo = enableRelativeTo(kRootFrame, this);

		var handle,
			mat_Handle = mat3(),
			aMat_Handle = [];

		var	handleRef = 0,
			length = this.aHandle.length,
			matLayerParent_Puppet = this.aMatLayerParent_Puppet[handleRef];

		// accumulate root matrix
		handle = this.aHandle[handleRef];
		getMatrixRelativeToParent(handle, mat_Handle);
		if (matLayerParent_Puppet) mat3.multiply(matLayerParent_Puppet, mat_Handle, mat_Handle);
		aMat_Handle.push(mat_Handle.clone());

		// walk downward (to leaves) to accumulate matrices
		while (++handleRef < length) {
			var parentRef = this.aParentRef[handleRef];

			matLayerParent_Puppet = this.aMatLayerParent_Puppet[handleRef];
			handle = this.aHandle[handleRef];
			getMatrixRelativeToParent(handle, mat_Handle);
			if (matLayerParent_Puppet) mat3.multiply(matLayerParent_Puppet, mat_Handle, mat_Handle);
			mat3.multiply(aMat_Handle[parentRef], mat_Handle, mat_Handle);
			aMat_Handle.push(mat_Handle.clone());
		}

		disableRelativeTo();

		return aMat_Handle;
	}

	function HandleTreeArray () {
		this.aHandle = [];
		this.aParentRef = [];
		this.aMatLayerParent_Puppet = [];
		this.aMatPuppet_LayerParent = [];
		this.size = 0;
		// TODO: use memoization for next two
		this.aIsLeaf = null;
		this.aLeafRef = null;
	}

	utils.mixin(HandleTreeArray, {
		getHandleRef : function (handle) {
			var handleRef = this.aHandle.indexOf(handle);
			if ( !isValidRef(handleRef) ) handleRef = null;

			return handleRef;
		},

		addLayer : function (layer, tagRef0) {
			var layerTree = layer.getHandleTreeRoot().gatherHandleTreeArray();

			arrayPush.apply(this.aHandle, layerTree.aHandle);

			var handleRef = 0, length = layerTree.aParentRef.length;

			// add root...
			if (isValidRef(tagRef0)) {
				this.aParentRef.push(tagRef0);
			} else {
				this.aParentRef.push(null);
			}
			this.aMatLayerParent_Puppet.push(layer.getSourceMatrixRelativeToLayer());
			this.aMatPuppet_LayerParent.push(layer.getLayerMatrixRelativeToSource());

			// and remaining descendents
			while (++handleRef < length) {
				this.aParentRef.push(this.size + layerTree.aParentRef[handleRef]);
				this.aMatLayerParent_Puppet.push(null);
				this.aMatPuppet_LayerParent.push(null);
			}

			this.size += length;
			this.aLeafRef = null;
			this.aIsLeaf = null;
		},

		getIsLeafArray : function () {
			if (this.aIsLeaf === null) {

				var aIsLeaf = [];

				aIsLeaf = fill(true, this.size);

				lodash.forEach(this.aParentRef, function (parentRef) {
					aIsLeaf[parentRef] = false;
				});

				this.aIsLeaf = aIsLeaf;
			}

			return this.aIsLeaf;
		},


		getLeafRefArray : function () {
			if (this.aLeafRef === null) {

				var aLeafRef = [],
					aIsLeaf = this.getIsLeafArray();

				lodash.forEach(aIsLeaf, function (isLeafRef, leafRef) {
					if (isLeafRef) aLeafRef.push(leafRef);
				});

				this.aLeafRef = aLeafRef;

			}

			return this.aLeafRef;
		},

		getAccumulatedHandleAtRest : function (kRootFrame, handleRef, result0) {
			return getAccumulatedHandleMatrix(this, kRootFrame, handleRef, "getMatrixAtRestRelativeToParent", result0);
		},

		getSourceFrames : function (kRootFrame) {
			var disableRelativeTo = enableRelativeTo(kRootFrame, this);

			var handle, 
				mat_Source = mat3(),
				aMat_Source = [],
				mat_Handle = mat3(),
				aMat_Handle = [];

			var	handleRef = 0,
				length = this.aHandle.length,
				matLayerParent_Puppet = this.aMatLayerParent_Puppet[handleRef];

			// accumulate root matrix
			handle = this.aHandle[handleRef];
			mat_Handle = getLocalMatrixAtRest(handle, mat_Handle);
			if (matLayerParent_Puppet) mat3.multiply(matLayerParent_Puppet, mat_Handle, mat_Handle);
			aMat_Handle.push(mat_Handle.clone());

			if (matLayerParent_Puppet) {
				mat3.initWithArray(mat_Handle, mat_Source);
			} else {
				mat_Source.setIdentity();
			}
			aMat_Source.push(mat_Source.clone());

			// walk downward (to leaves) to accumulate matrices
			while (++handleRef < length) {
				var parentRef = this.aParentRef[handleRef];

				matLayerParent_Puppet = this.aMatLayerParent_Puppet[handleRef];
				handle = this.aHandle[handleRef];
				getLocalMatrixAtRest(handle, mat_Handle);
				if (matLayerParent_Puppet) mat3.multiply(matLayerParent_Puppet, mat_Handle, mat_Handle);
				mat3.multiply(aMat_Handle[parentRef], mat_Handle, mat_Handle);
				aMat_Handle.push(mat_Handle.clone());

				if (matLayerParent_Puppet) {
					mat3.multiply(aMat_Handle[parentRef], matLayerParent_Puppet, mat_Source);
				} else {
					mat3.initWithArray(aMat_Source[parentRef], mat_Source);
				}
				aMat_Source.push(mat_Source.clone());
			}

			disableRelativeTo();

			return aMat_Source;
		},

		getAccumulatedTreeAtRest : function (kRootFrame) {
			return getAccumulatedTree.call(this, kRootFrame, getLocalMatrixAtRest);
		},

		getParentRef : function (handleRef) {
			var parentRef = this.aParentRef[handleRef];
			if ( isValidRef(parentRef) ) return parentRef;

			return null;
		}
		
	});

	return HandleTreeArray;
});
